function [d_stks,psf] = deconv_stk2(varargin)
%little function that applies a 2d deconvolution to a stack of images.
%Synatax:   [d_stk,psf] = deconv_stk('edgetaper',1,'deconv','reg','numit',10,...
%               'initpsf',10); 
%Input:     'edgetaper' = taper the image edges according to the PSF.
%               Default = 0; off.
%           'deconv' = type of deconvolution used.  Default = lucy
%               Lucy-Richardson.  Other types: 'reg' = regularized,
%               'wiener' = wiener, and 'blind' = blind, based on maximum
%               likelihood algorithm.
%           'numit' = number of iteration for deconvolution, default = 10.
%               only applicable to blind and Lucy-Richardson.
%           'dampar' = the threshold deviation of the resulting image from 
%               the original image (in terms of the standard deviation of 
%               Poisson noise) below which damping occurs
%           'subsmpl' =  denotes subsampling and is used when the PSF is
%               given on a grid that is SUBSMPL times finer than the image. 
%               Default is 1.
%           'initpsf' = the size of the initial psf generated for the blind
%               deconvolution.  The PSF restoration is affected strongly by
%               the size of the initial guess INITPSF and less by the 
%               values it contains.
%           'cfx' = change file data types to preserve flux ratios. Default
%               = 0, off.  In this case, a 8 bit image will go to 16 and
%               back to 8bit, 16 bit image will go to 32bit then back to 16
%               for storage.
%           'save' = save the stacks as tiff files automatically. Default =
%               1.  0 = off.
%           'multi' = multithreading stacks.  Default = 0 or off.
%Output:    d_stk = the deconvoluted image stack
%           psf = the PSF used.
%           fac = the psf parameter used. (only if lucy is used)

[psf,edgetapered,deconv_type,numit,dampar,subsmpl,sav,stks_ch_names,channel_paths,filterindex,cfx,multi] = parse(varargin);

%h2 = waitbar(0,'Image Stack: ','position',[20 300 275 50]);    %initialize progress bar.
d_stks = cell(1,size(stks_ch_names,2));

%localization for macs and linux
if ispc
    slash = '\';        %Windows directory marker
else
    slash = '/';        %Mac directory marker
end

for i = 1:size(stks_ch_names,2)      %process each image stack
    %open the stack, no need to cell mode
    [stks] = stack_gui('filename',stks_ch_names{1,i},'pathname',channel_paths,'filterindex',filterindex);
    %make sure dampar is the correct type
    imgclass = class(stks);      %get image type
    switch imgclass
        case 'uint8'
            dampar = uint8(dampar);      %the image is unsigned 8bit
            if cfx      %preserve flux
                stks = uint16(stks);
                imgclass = 'uint16';
                dampar = uint16(dampar);
            end
        case 'uint16'
            dampar = uint16(dampar);     %the image is unsigned 16bit
            if cfx
                stks = im2double2(uint32(stks));
                imgclass = 'uint32b';
                dampar = double(dampar);
            end
        case 'uint32'                    %no support for 32bit yet, convert to double
            %dampar = uint32(dampar);    
            stks = im2double2(stks);
            dampar = double(dampar);
        case 'double'
            dampar = double(dampar);     %the image is double
        case 'single'
            dampar = single(dampar);     %the image is single
    end
    %parallelize
    if multi
        parpool      %initiate processes
    end
    %waitbar(i/size(stks,2),h2,['Image Stack: ',stks_ch_names{i}]);   %update progress
    if edgetapered        %if the edges need to be tapered
        stk_tmp = edgetaper(stks,psf);
    else
        stk_tmp = stks;
    end
    %initiate temporary stack holder
    if strcmp('uint32',imgclass)||strcmp('uint32b',imgclass)    %32 for now is double
        stk_out = zeros(size(stk_tmp),'double');
    else
        stk_out = zeros(size(stk_tmp),imgclass);
    end
    %now apply the deconvolution
    switch deconv_type
        case 'lucy'
            if multi    %if going for multi threads
                parfor j = 1:size(stk_tmp,3)
                    tic
                    stk_out(:,:,j) = deconvlucy(stk_tmp(:,:,j),psf,numit,dampar,[],[],subsmpl);
                    toc
                end
            else        %single thread only
                h = waitbar(0,['Deconvolving Slice (',stks_ch_names{1,i}(1:end-4),'): 1']);    %initialize progress bar.
                parfor j = 1:size(stk_tmp,3)
                    stk_out(:,:,j) = deconvlucy(stk_tmp(:,:,j),psf,numit,dampar,[],[],subsmpl);
                    waitbar(j/size(stk_tmp,3),h,['Deconvolving Slice (',stks_ch_names{1,i}(1:end-4),'): ',num2str(j+1)]);   %update progress
                end
                close(h)
            end
        case 'reg'
            %h = waitbar(0,'Deconvolving Slice: 1');    %initialize progress bar.
            parfor j = 1:size(stk_tmp,3)
                tic
                stk_out(:,:,j) = deconvreg(stk_tmp(:,:,j),psf);
                %waitbar(j/size(stk_tmp,3),h,['Deconvolving Slice: ',num2str(j+1)]);   %update progress
                toc
            end
            %close(h)
        case 'wiener'
            %h = waitbar(0,'Deconvolving Slice: 1');    %initialize progress bar.
            parfor j = 1:size(stk_tmp,3)
                tic
                stk_out(:,:,j) = deconvwnr(stk_tmp(:,:,j),psf);
                %waitbar(j/size(stk_tmp,3),h,['Deconvolving Slice: ',num2str(j+1)]);   %update progress
                toc
            end
            %close(h)
        case 'blind'
            initpsf = psf;      %psf is the output so it will change, temp the PSF.
            %h = waitbar(0,'Deconvolving Slice: 1');    %initialize progress bar.
            parfor j = 1:size(stk_tmp,3)
                tic
                [stk_out(:,:,j),psf(:,:,j)] = deconvblind(stk_tmp(:,:,j),initpsf,numit,dampar);
                %waitbar(j/size(stk_tmp,3),h,['Deconvolving Slice: ',num2str(j+1)]);   %update progress
                toc
            end
            %close(h)
    end
    %stretch dynamic range (decomment if you want it on)
    %stk_out = imnorm(stk_out);
    %if cfx is on change the data back to the initial data type
    if cfx
        switch imgclass
            case 'uint32'                    %32bit in 32bit out
                stk_out = uint32(stk_out.*4294967295);
            case 'uint32b'                    %16bit in 16bit out a special case I guess
                d2_stks{1,i} = im2uint16(imnorm(stk_out));          %output the data in 16bit for compatibility and ease of use
                stk_out = uint32(stk_out.*4294967295);      %output the full image   
            otherwise
                stk_out = im2uint16(stk_out);   %uint16 is default
        end
    else  %otherwise make sure the output is uint16 or unint32 (comment out if you do not want this)
        switch imgclass
            case 'uint32'                    %32bit in 32bit out
                stk_out = uint32(stk_out.*4294967295);
            otherwise
                stk_out = im2uint16(stk_out);   %uint16 is default
        end
    end
    d_stks{1,i} = stk_out;      %output deconvoluted stack
    clear stk_tmp
    if multi
        delete(gcp('nocreate')) %close processes
    end
    %save the stacks automatically if selected.
    if sav
        mkdir(channel_paths,'deconv');
        mkdir([channel_paths,'deconv'],stks_ch_names{1,i}(1:end-4));
        stk2tiff(d_stks{1,i},stks_ch_names{1,i},[channel_paths,'deconv',slash,stks_ch_names{1,i}(1:end-4),slash]);
        if cfx&&strcmp('uint32b',imgclass)
            mkdir([channel_paths,'deconv'],[stks_ch_names{1,i}(1:end-4),'_16bit']);
            stk2tiff(d2_stks{1,i},stks_ch_names{1,i},[channel_paths,'deconv',slash,stks_ch_names{1,i}(1:end-4),'_16bit',slash]);
        end
    end
end
%close(h2)




%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [psf,edgetapered,deconv_type,numit,dampar,subsmpl,sav,stks_ch_names,channel_paths,filterindex,cfx,multi] = parse(input)

deconv_type = 'lucy';    %1=lucy-richardson(default), 2=regularized, 3=wiener, 4=blind.
edgetapered = 0;  %off.
numit = 10;     %default # of iteration = 10.
initpsf = 10;  %default = 10x10
sav = 1;       %default = save the file automatically
dampar = 0;     %default = 0 no dampling
subsmpl = 1;    %deault = 1 no subsampling
cfx = 0;     %by default off
multi = 0;      %multi on by default

%open the image stack first.  Prompt first
prompt_box('title','Open Image Stack','prompt1','Select image stack or stacks ',...
    'prompt2','for deconvolution','position','center');
pause(0.25)
%Get the location of the stacks you want to open
[stks_ch_names,channel_paths,filterindex] = uigetfile2b({'*.tif','TIFF files (*.tif)';...
    '*.jpg','Jpeg files (*.jpg)';'*.gif','GIF files (*.gif)';...
    '*.bmp','Bitmap files (*.bmp)';'*.mat','Mat files (*.mat)';},'Open Stack','Multiselect','on');

%make sure the file names are in a cell array
if ~iscell(stks_ch_names)
    stks_ch_names = {stks_ch_names};
end

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'edgetaper'
                    edgetapered = input{1,i+1};
                case 'deconv'
                    if ischar(input{1,i+1})
                        deconv_type = input{1,i+1};
                    else
                        warning(['Your entered deconvolution type is not recognized, reverting to defalut type.']);
                    end
                case 'numit'
                    numit = input{1,i+1};
                case 'save'
                    sav = input{1,i+1};
                case 'dampar'
                    dampar = input{1,i+1};
                case 'subsmpl'
                    subsmpl = input{1,i+1};
                case 'initpsf'
                    initpsf = input{1,i+1};
                case 'cfx'
                    cfx = input{1,i+1};
                case 'multi'
                    multi = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end

%open the PSF image.  Prompt first
if ~strcmp(deconv_type,'blind')          %if the deconvolution is not blind open PSF
    prompt_box('title','Import the PSF','prompt1','Select the desired PSF ',...
        'prompt2','PSF should be in image form','position','center');
    pause(0.25)
    [psf] = stack_gui(1);  %use cell output mode.
    psf = im2double(psf{1,1});       %the PSF needs to be in double percision.
    %matlab expect the PSF to sum to 1, thus lets do a little math to check
    psf = psf./(sum(sum(psf)));
else                                %if the deconvolution is blind generate PSF
    psf = fspecial('gaussian',initpsf,10);
end